//////////////////////////////////////////////////////////////////////////////
//
//  Copyright 2017 Autodesk, Inc.  All rights reserved.
//
//  Use of this software is subject to the terms of the Autodesk license 
//  agreement provided at the time of installation or download, or which 
//  otherwise accompanies this software in either electronic or hard copy form.   
//
//////////////////////////////////////////////////////////////////////////////

#include "MAXtoATestPlugin.h"
#include "MAXtoATestLight.h"

// arnold
#undef min
#undef max
#include <ai.h>

#include <ai_nodes.h>
#include <ai_array.h>

#include "resource.h"
#include <iparamb2.h>
#include <macrorec.h>

#include "../../include/3dsmax_banned.h"

#ifndef M_PI        /* Pi */
#define M_PI        3.14159265358979323846
#endif

class MAXtoATestPlugin_ClassDesc : public ClassDesc2 
{
public:
    MAXtoATestPlugin_ClassDesc();

    virtual int IsPublic() override;
    virtual void* Create(BOOL loading) override;
    virtual const TCHAR* ClassName() override;
    virtual SClass_ID SuperClassID() override;
    virtual Class_ID ClassID() override;
    virtual const TCHAR* Category() override;
    virtual const TCHAR* InternalName() override;
    virtual HINSTANCE HInstance() override;

protected:
    virtual MaxSDK::QMaxParamBlockWidget* CreateQtWidget(
        ReferenceMaker& owner,
        IParamBlock2& paramBlock,
        const MapID paramMapID,
        MSTR& rollupTitle,
        int& rollupFlags,
        int& rollupCategory) override;

private:
    // Prefernces change callback
    static void PreferencesChangeCallback(void* param);
    bool m_preferenceCallbackRegistered;
};

MAXtoATestPlugin_ClassDesc::MAXtoATestPlugin_ClassDesc()
: m_preferenceCallbackRegistered(false)
{
}

void MAXtoATestPlugin_ClassDesc::PreferencesChangeCallback(void* param) 
{
    MAXtoATestPlugin_ClassDesc* classDesc = static_cast<MAXtoATestPlugin_ClassDesc*>(param);
    if(DbgVerify(classDesc != nullptr))
    {
        ClassEntry* classEntry = GetCOREInterface()->GetDllDir().ClassDir().FindClassEntry(classDesc->SuperClassID(), classDesc->ClassID());
        DbgAssert(classEntry != NULL);
        if (classEntry != NULL) 
        {
            GetCOREInterface()->DeleteClass(classDesc);
            GetCOREInterface()->AddClass(classDesc);
        }
    }
}

int MAXtoATestPlugin_ClassDesc::IsPublic() 
{
    return TRUE;
}

void* MAXtoATestPlugin_ClassDesc::Create(BOOL loading) 
{
    return new MAXtoATestPlugin(loading != 0);
}

const TCHAR* MAXtoATestPlugin_ClassDesc::ClassName() 
{
    static const MSTR str = L"MAXtoATest Light";
    return str;
}

SClass_ID MAXtoATestPlugin_ClassDesc::SuperClassID() 
{
    return LIGHT_CLASS_ID;
}

Class_ID MAXtoATestPlugin_ClassDesc::ClassID() 
{
    return MAXtoATestPLUGIN_CLASS_ID;
}

const TCHAR* MAXtoATestPlugin_ClassDesc::Category() 
{
    static const MSTR str = L"MAXtoATest";
    return str;
}

const TCHAR* MAXtoATestPlugin_ClassDesc::InternalName() 
{
    return _T("MAXtoATest Light");
}

HINSTANCE MAXtoATestPlugin_ClassDesc::HInstance() 
{
    extern HINSTANCE hInstance;
    return hInstance;
}

MaxSDK::QMaxParamBlockWidget* MAXtoATestPlugin_ClassDesc::CreateQtWidget(
    ReferenceMaker& owner,
    IParamBlock2& paramBlock,
    const MapID paramMapID,
    MSTR& rollupTitle,
    int& rollupFlags,
    int& rollupCategory)
{
    switch (paramMapID)
    {
        case 0:
        {
            rollupTitle = L"MAXtoATest Light";
            MAXtoATestLight* const widget = new MAXtoATestLight(owner, paramBlock);
            rollupCategory -= 70;
            return widget;
        }
    }

    return nullptr;
}

//==============================================================================
// class ArnoldLightObject
//==============================================================================

// The param block descriptor
ParamBlockDesc2 MAXtoATestPlugin::m_mainPBDesc(
    0,
    _T("Parameters"), 
    IDS_PARAMETERS, 
    &MAXtoATestPlugin::GetClassDesc(),
    P_AUTO_CONSTRUCT | P_AUTO_UI_QT | P_MULTIMAP,

    0,

    1, // num param maps
    0, 

    kMainParamID_On, _T("on"), TYPE_BOOL, 0, IDS_ON,
    p_default, true,
    p_end,

    kMainParamID_Map, _T("map"), TYPE_TEXMAP, 0, IDS_MAP,
    p_end,

p_end
);

ClassDesc2& MAXtoATestPlugin::GetClassDesc() 
{
    static MAXtoATestPlugin_ClassDesc classDesc;
    return classDesc;
}

MAXtoATestPlugin::MAXtoATestPlugin(bool loading)
    : m_ti(this)
{
    if (!loading)
        GetClassDesc().MakeAutoParamBlocks(this);
}

MAXtoATestPlugin::~MAXtoATestPlugin()
{
    DeleteAllRefs();
}

void* MAXtoATestPlugin::GetInterface(ULONG id)
{
    return LightObject::GetInterface(id);
}

void MAXtoATestPlugin::DeleteThis()
{
    delete this;
}

void MAXtoATestPlugin::GetClassName(TSTR& s)
{
    s = L"MAXtoATest Light";
}

BOOL MAXtoATestPlugin::IsSubClassOf(Class_ID classID)
{
    return classID == MAXtoATestPLUGIN_CLASS_ID;
}

int MAXtoATestPlugin::NumSubs()
{
    return 1;
}

Animatable* MAXtoATestPlugin::SubAnim(int i)
{
    switch (i)
    {
    case 0:
        return m_mainPB;
    default:
        return nullptr;
    }
}

TSTR MAXtoATestPlugin::SubAnimName(int i)
{
    switch (i)
    {
    case 0:
        return MaxSDK::GetResourceStringAsMSTR(IDS_PARAMETERS);
    default:
        return TSTR();
    }
}

void MAXtoATestPlugin::InitNodeName(MSTR& s)
{ 
    s = _M("MAXtoATestLight"); 
}

void MAXtoATestPlugin::buildMesh(Mesh* mesh, Point3 center) const
{
    int lat = 8;
    int lon = 8;

    mesh->setNumVerts((lat+1)*lon);
    mesh->setNumFaces(lat*lon);

    int v = 0;
    for (int wLon = 0; wLon < lon; ++wLon)
    {
        for (int wLat = 0; wLat <= lat; ++wLat)
        {
            Point3 vert = 8.f*Point3(sinf(M_PI * (float)wLat / (float)lat) * cosf(2.f * (float)M_PI * (float)wLon / (float)lon),
                                     sinf(M_PI * (float)wLat / (float)lat) * sinf(2.f * M_PI * (float)wLon / (float)lon),
                                     cosf((float)M_PI * (float)wLat / (float)lat));
            mesh->setVert(v++, vert);
        }
    }

    int limit = lat * lon - 1;
    int f = 0;
    int c = lat;
    for (int i = 0; i < limit; ++i)
    {
        if (i == c)
        {
            c += lat + 1;
            continue;
        }

        mesh->faces[f].setVerts(i, i + 1, i + lat + 2);
        mesh->faces[f].setSmGroup(0);
        mesh->faces[f++].setEdgeVisFlags(1, 1, 0);
    }

    for (int i = 0; i < lat; ++i)
    {
        mesh->faces[f].setVerts(i + limit, i + 1 + limit, i + 1);
        mesh->faces[f].setSmGroup(0);
        mesh->faces[f++].setEdgeVisFlags(1, 1, 0);
    }
}

// placeholders

RefTargetHandle	MAXtoATestPlugin::GetReference(int i)
{
    switch (i)
    {
    case 0:
        return m_mainPB;
    default:
        DbgAssert(false);
        return nullptr;
    }
}

void MAXtoATestPlugin::SetReference(int i, RefTargetHandle rtarg)
{
    switch (i)
    {
    case 0:
        m_mainPB = dynamic_cast<IParamBlock2*>(rtarg);
        DbgAssert(m_mainPB == rtarg);
        break;
    default:
        DbgAssert(false);
        break;
    }
}

RefTargetHandle	MAXtoATestPlugin::Clone(RemapDir &remap)
{
    MAXtoATestPlugin* newLight = new MAXtoATestPlugin(false);

    // First clone the parameter block
    const int count = NumRefs();
    for (int i = 0; i < count; ++i) 
    {
        ReferenceTarget* refTarg = GetReference(i);
        if (refTarg != NULL)
            newLight->ReplaceReference(i, remap.CloneRef(refTarg));
    }

    BaseClone(this, newLight, remap);
    return (RefTargetHandle)newLight;
}

Point3 MAXtoATestPlugin::GetRGBColor(TimeValue t, Interval &valid)
{
    return Point3(1, 1, 1);
}

void MAXtoATestPlugin::BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev)
{
    GetClassDesc().BeginEditParams(ip, this, flags, prev);
}

void MAXtoATestPlugin::EndEditParams(IObjParam *ip, ULONG flags, Animatable *next)
{
    GetClassDesc().EndEditParams(ip, this, flags, next);
}

IParamBlock2* MAXtoATestPlugin::GetParamBlock(int i)
{
    switch (i)
    {
    case 0:
        return m_mainPB;
    default:
        DbgAssert(false);
        return nullptr;
    }
}

IParamBlock2* MAXtoATestPlugin::GetParamBlockByID(BlockID id)
{
    switch (id)
    {
    case 0:
        return m_mainPB;
    default:
        return nullptr;
    }
}

RefResult MAXtoATestPlugin::NotifyRefChanged(const Interval &validity, RefTargetHandle hTarget, PartID &, RefMessage message, BOOL)
{
    return REF_SUCCEED;
}

void MAXtoATestPlugin::NotifyTarget(int message, ReferenceMaker* hMaker)
{
    if (message == TARGETMSG_TRANSFORMING_NODE)
    {
    }
}

INode* MAXtoATestPlugin::GetNodeRef(ReferenceMaker *rm)
{
    if (rm->SuperClassID() == BASENODE_CLASS_ID)
        return (INode *)rm;

    if (rm->IsRefTarget())
    {
        DependentIterator di(static_cast< ReferenceTarget* >(rm));
        ReferenceMaker *rm2 = NULL;
        INode *nd = NULL;

        while ((rm2 = di.Next()) != NULL) 
        {
            nd = GetNodeRef(rm2);
            if (nd)
                return nd;
        }

        return NULL;
    }

    return nullptr;
}

ObjectState MAXtoATestPlugin::Eval(TimeValue)
{
    return ObjectState(this);
}

RefResult MAXtoATestPlugin::EvalLightState(TimeValue t, Interval& valid, LightState* ls)
{
    return REF_SUCCEED;
}

void MAXtoATestPlugin::SetUseLight(int onOff)
{
    m_mainPB->SetValue(kMainParamID_On, 0, onOff);
    NotifyDependents(FOREVER, PART_OBJ, REFMSG_CHANGE);
}

BOOL MAXtoATestPlugin::GetUseLight(void)
{
    BOOL onOff;
    Interval valid = FOREVER;
    m_mainPB->GetValue(kMainParamID_On, 0, onOff, valid);
    return onOff;
}

void MAXtoATestPlugin::SetHotspot(TimeValue, float)
{
    // not implemented
}

float MAXtoATestPlugin::GetHotspot(TimeValue, Interval &)
{
    return -1.f;
}

void MAXtoATestPlugin::SetFallsize(TimeValue, float)
{
    // not implemented
}

float MAXtoATestPlugin::GetFallsize(TimeValue, Interval &)
{
    return -1.f;
}

void MAXtoATestPlugin::SetAtten(TimeValue, int, float)
{
    // not implemented
}

float MAXtoATestPlugin::GetAtten(TimeValue, int, Interval &)
{
    return 0.f;
}

void MAXtoATestPlugin::SetTDist(TimeValue, float)
{
    // not implemented
}

float MAXtoATestPlugin::GetTDist(TimeValue, Interval &)
{
    return 0.f;
}

void MAXtoATestPlugin::SetConeDisplay(int, int)
{
    // not implemented
}

BOOL MAXtoATestPlugin::GetConeDisplay(void)
{
    return true;
}

void MAXtoATestPlugin::SetExtendedDisplay(int flags)
{
    m_extDispFlags = flags;
}

/* From BaseObject */

enum HitFlags 
{
    HIT_SHAPE = 1,
    HIT_LINE = 2
};

int MAXtoATestPlugin::HitTest(TimeValue t, INode *inode, int type, int crossing, int flags, IPoint2 *p, ViewExp *vpt)
{
    if (!vpt || !vpt->IsAlive())
    {
        DbgAssert(!_T("Invalid viewport!"));
        return FALSE;
    }

    int result = false;
    DWORD savedLimits;

    GraphicsWindow *gw = vpt->getGW();

    {
        HitRegion hitRegion;
        MakeHitRegion(hitRegion, type, crossing, 4, p);
        gw->setRndLimits(((savedLimits = gw->getRndLimits()) | GW_PICK) & ~(GW_ILLUM | GW_BACKCULL));
        Matrix3 m;
        m = inode->GetObjectTM(t);
        m.NoScale();
        float scaleFactor = vpt->NonScalingObjectSize() * vpt->GetVPWorldWidth(m.GetTrans()) / 360.0f;
        m.Scale(Point3(scaleFactor, scaleFactor, scaleFactor));
        gw->setTransform(m);

        gw->clearHitCode();

        result = m_mesh.select(gw, gw->getMaterial(), &hitRegion, flags & HIT_ABORTONHIT);
        if (!result) 
        {
            result = Draw(t, *vpt, *gw, *inode, flags, true);

            if (result & HIT_LINE)
                inode->SetTargetNodePair(1);
        }
        gw->setRndLimits(savedLimits);
    }

    return result;
}

const MCHAR* MAXtoATestPlugin::GetObjectName()
{
    return L"MAXtoATest Light";
}

int MAXtoATestPlugin::Draw(const TimeValue t, ViewExp& vpt, GraphicsWindow& gw, INode& node, const int flags, bool hitTest)
{
    int result = 0;

    const DWORD rlim = gw.getRndLimits();
    {
        gw.clearHitCode();

        bool isSelected = node.Selected() != 0;

        if (!m_meshBuilt)
        {
            Point3 pos(0.0f, 0.0f, 0.0f);
            buildMesh(&m_mesh, pos);
            m_meshBuilt = true;
        }

        Color wireframe_color = Color(node.GetWireColor());
        if (isSelected)
            wireframe_color =  GetSelColor();
        else if (!node.IsFrozen() && !node.Dependent())
            wireframe_color = GetUseLight() ? Color(node.GetWireColor()) : Color(0.0f, 0.0f, 0.0f);
        else if (node.IsFrozen())
            wireframe_color = GetFreezeColor();

        gw.setColor(LINE_COLOR, wireframe_color);

        {
            Matrix3 tm = node.GetObjectTM(t);
            tm.NoScale();
            float scaleFactor = vpt.NonScalingObjectSize() * vpt.GetVPWorldWidth(tm.GetTrans()) / 360.0f;
            tm.Scale(Point3(scaleFactor, scaleFactor, scaleFactor));
            gw.setTransform(tm);
            m_mesh.render(&gw, gw.getMaterial(), (flags&USE_DAMAGE_RECT) ? &vpt.GetDammageRect() : NULL, COMP_ALL);
        }

        {
            gw.clearHitCode();
            Matrix3 tm = node.GetObjectTM(t);
            gw.setTransform(tm);
            
            if (isSelected)
                DrawShape(t, vpt, gw, node, flags, wireframe_color);

            result = gw.checkHitCode();
            gw.clearHitCode();
        }
    }
    gw.setRndLimits(rlim);

    return result;
}

void MAXtoATestPlugin::DrawShape(const TimeValue t, ViewExp& vpt, GraphicsWindow& gw, INode& node, const int flags, const Color& color)
{
}


int MAXtoATestPlugin::Display(TimeValue t, INode* inode, ViewExp* vpt, int flags)
{

    if (DbgVerify((vpt != nullptr) && vpt->IsAlive() && (inode != nullptr)))
    {
        GraphicsWindow *gw = vpt->getGW();
        DWORD savedLimits = gw->getRndLimits();
        gw->setRndLimits(GW_WIREFRAME | GW_EDGES_ONLY | (gw->getRndMode() & GW_Z_BUFFER));
        Draw(t, *vpt, *gw, *inode, flags, false);
        gw->setRndLimits(savedLimits);
    }

    return 0;
}


IOResult MAXtoATestPlugin::Load(ILoad *iload)
{
    for (IOResult open_chunk_res = iload->OpenChunk(); open_chunk_res == IO_OK; open_chunk_res = iload->OpenChunk())
    {
        IOResult res = IO_OK;
        const int id = iload->CurChunkID();
/*
        switch (id)
        {
        //case IOChunkID_WeatherFileData:
        //	res = m_weather_file_ui_data.Load(iload);
        //	break;
        default:
            break;
        }
*/

        iload->CloseChunk();
        if (res != IO_OK)
            return res;
    }

    return IO_OK;
}

IOResult MAXtoATestPlugin::Save(ISave *isave)
{
    if (isave != nullptr)
    {
        //isave->BeginChunk(IOChunkID_WeatherFileData);
        //const IOResult res = m_weather_file_ui_data.Save(isave);
        //if (res != IO_OK)
        //	return res;
        //isave->EndChunk();
    }

    return IO_OK;
}

void MAXtoATestPlugin::GetWorldBoundBox(TimeValue t, INode* inode, ViewExp* vpt, Box3& box)
{
    Interval dummy_interval;
    Matrix3 tm(1);
    tm = inode->GetObjectTM(t);
    box = m_mesh.getBoundingBox(&tm);

    if (!(m_extDispFlags & EXT_DISP_GROUP_EXT))
    {
        Point3 loc = inode->GetObjectTM(t).GetTrans();
        Matrix3 tmat;
        inode->GetTargetTM(t, tmat);
        Point3 pt = tmat.GetTrans();
        float d = FLength(loc - pt) / FLength(inode->GetObjectTM(t).GetRow(2));
        box += (tm)*Point3(0.0f, 0.0f, -d);
    }

    /*
    if (MaxSDK::Graphics::IsRetainedModeEnabled() && !(m_extDispFlags & EXT_DISP_ZOOM_EXT))
    {
        #define ENLARGE_FACTOR 3.0f
        float scaleFactor = vpt->NonScalingObjectSize()
            * vpt->GetVPWorldWidth(tm.GetTrans()) / 360.0f;
        if (scaleFactor < ENLARGE_FACTOR)
            box.EnlargeBy(ENLARGE_FACTOR / scaleFactor);
    }
    */
}

void MAXtoATestPlugin::GetLocalBoundBox(TimeValue t, INode* inode, ViewExp* vpt, Box3& box)
{
    Interval dummy_interval;
    Matrix3 tm(1);
    box = m_mesh.getBoundingBox(&tm);
}

void MAXtoATestPlugin::GetDeformBBox(TimeValue t,Box3& box, Matrix3 *tm, BOOL useSel)
{
    box = m_mesh.getBoundingBox(tm);
}

//==============================================================================
// class MAXtoATestPlugin_CreateCallback
//==============================================================================

MAXtoATestPlugin_CreateCallback MAXtoATestPlugin_CreateCallback::m_theInstance;

MAXtoATestPlugin_CreateCallback::MAXtoATestPlugin_CreateCallback()
{
}

int MAXtoATestPlugin_CreateCallback::proc(ViewExp *vpt, int msg, int point, int flags, IPoint2 m, Matrix3& mat)
{
    if (msg == MOUSE_POINT)
    {
        if (point == 0)
        {
#ifdef _3D_CREATE
            mat.SetTrans(vpt->SnapPoint(m, m, NULL, SNAP_IN_3D));
#else
            mat.SetTrans(vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE));
#endif
            return CREATE_STOP;
        }
    }
    
    else if (msg == MOUSE_ABORT)
        return CREATE_ABORT;

    return CREATE_CONTINUE;
}

// MAXtoATestPlugin callback mouse method
CreateMouseCallBack* MAXtoATestPlugin::GetCreateMouseCallBack(void)
{
    MAXtoATestPlugin_CreateCallback& callback = MAXtoATestPlugin_CreateCallback::GetInstance();
    return &callback;
}

void MAXtoATestPlugin::EnumAuxFiles(AssetEnumCallback &nameEnum, DWORD flags)
{
}

Object* MAXtoATestPlugin::MakeShallowCopy(ChannelMask channels)
{
    return this;
}

BaseInterface* MAXtoATestPlugin::GetInterface(Interface_ID id)
{
    if (id == MAXTOA_TranslationInterface_ID)
        return &(this->m_ti);

    return LightObject::GetInterface(id);
}


MAXtoATestPlugin_translationInterface::MAXtoATestPlugin_translationInterface(MAXtoATestPlugin* plugin)
{
    m_plugin = plugin;
}

namespace
{
    void MaxToArnold(const Matrix3 &in_m, AtMatrix &out_m)
    {
        out_m[0][0] = in_m[0][0]; out_m[0][1] = in_m[0][1]; out_m[0][2] = in_m[0][2]; out_m[0][3] = 0.0f;
        out_m[1][0] = in_m[1][0]; out_m[1][1] = in_m[1][1]; out_m[1][2] = in_m[1][2]; out_m[1][3] = 0.0f;
        out_m[2][0] = in_m[2][0]; out_m[2][1] = in_m[2][1]; out_m[2][2] = in_m[2][2]; out_m[2][3] = 0.0f;
        out_m[3][0] = in_m[3][0]; out_m[3][1] = in_m[3][1]; out_m[3][2] = in_m[3][2]; out_m[3][3] = 1.0f;
    }
}

Arnold_translation::TranslationInterface_Result MAXtoATestPlugin_translationInterface::Translate( INode* node,
                                                    Arnold_translation* arnold_translation,
                                                    const TimeValue translationTime,
                                                    Interval& newValidity)
{
    Arnold_translation::TranslationInterface_Result res = Arnold_translation::Success;

    AtNode* aNode = arnold_translation->GetOutput(0, AtString("quad_light"), AtString("MyMAXtoATestLight"));

    AiNodeSetFlt(aNode, "intensity", 1000.f);

    AtArray* vertices = AiArrayAllocate(4, 1, AI_TYPE_VECTOR);
    AiArraySetVec(vertices, 0, AtVector(10, -10, 0));
    AiArraySetVec(vertices, 1, AtVector(-10, -10, 0));
    AiArraySetVec(vertices, 2, AtVector(-10, 10, 0));
    AiArraySetVec(vertices, 3, AtVector(10, 10, 0));
    AiNodeSetArray(aNode, "vertices", vertices);

    res = arnold_translation->PerformGenericTranslation(0, 0);
    
    bool isColorSet = false;

    Texmap *map = m_plugin->m_mainPB->GetTexmap(MAXtoATestPlugin::kMainParamID_Map, 0);
    if (map)
    {
        AtNode* mNode = arnold_translation->GetNode(Arnold_translation::NodeType_Texmap, map);
        if (mNode != nullptr)
        {
            AiNodeLink(mNode, "color", aNode);
            isColorSet = true;
        }
    }
    
    if (!isColorSet)
        AiNodeSetRGB(aNode, "color", 0.f, 0.f, 1.f);

    return res;
}

Arnold_translation::TranslationInterface_Result MAXtoATestPlugin_translationInterface::TranslateKeyframe(INode* node,
    const TimeValue frame_time,
    const TimeValue keyframe_time,
    unsigned int keyframe_index)
{
    return Arnold_translation::Success;
}
